//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
// SIO4 Example Code
//------------------------------------------------------------------------------
// Revision History:
//------------------------------------------------------------------------------
// Revision  Date        Name      Comments
// 1.0       08/29/03    Gary      Released
//
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <conio.h>
#include <math.h>
#include <time.h>
#include "ciocolor.h"
#include "tools.h"
#include "SIO4intface.h"


/*****************************************************/
/* Forward declarations of local functions           */
/*****************************************************/
void ResetBoard(void);
void WriteLocal(void);
void ReadLocal(void);
void WriteUSC(void);
void ReadUSC(void);
void WriteConfig(void);
void ReadConfig(void);
void InternalLoopback(void);
// test functions
void SingleChannelInternalLoopbackTest(U32 ulChannel);

/*****************************************************/
/* Global Vars                                       */
/*****************************************************/
// read and write data buffers:
U32	*pReadBuffer  = (U32 *) NULL;
U32	*pWriteBuffer = (U32 *) NULL;

HGLOBAL         hWriteGlobal;
HGLOBAL         hReadGlobal;

U32	ulBufferSize = 0x100;

U32 ulNumBds, ulBdNum, ulAuxBdNum, ulErr = 0;
U16 saved_x, saved_y, CurX=2, CurY=2;
char mmchar, kbchar, datebuf[16], timebuf[16];
char cBoardInfo[400];


//==============================================================================
//
//==============================================================================
void main(int argc, char * argv[])
{
  CursorVisible(FALSE);

  /*****************************************************/
  /* Get two buffers for data: one read, one write     */
  /* GlobalAlloc expects memory size in bytes          */
  /*****************************************************/
	hWriteGlobal = GlobalAlloc(GMEM_FIXED, ulBufferSize*4);
	if (! hWriteGlobal)
		printf("\n\t\tFailed to get write buffer\n");

	hReadGlobal  = GlobalAlloc(GMEM_FIXED, ulBufferSize*4);
	if (! hReadGlobal)
		printf("\n\t\tFailed to get read buffer\n");

	if ((hWriteGlobal) && (hReadGlobal))
	{
		pWriteBuffer = (U32 *) GlobalLock(hWriteGlobal);
		pReadBuffer  = (U32 *) GlobalLock(hReadGlobal);
	}

	// unable to get either ReadBuffer or WriteBuffer - end and exit
	if ((! pWriteBuffer) || (! pReadBuffer))
	{
		printf("\n\t\tCannot Allocate Buffer Memory. Hit Any Key to exit...");
		getch();
		if (hWriteGlobal){GlobalUnlock(hWriteGlobal);}
		if (pWriteBuffer){GlobalFree(hWriteGlobal);}
		if (hReadGlobal){GlobalUnlock(hReadGlobal);}
		if (pReadBuffer){GlobalFree(hReadGlobal);}
		exit(0);
	}

  ulNumBds = SIO4_FindBoards(&cBoardInfo[0], &ulErr);

  if(ulErr)
  {
   ShowAPIError(ulErr);
   do{}while( !kbhit() );         // Wait for a Key Press
   exit(0);
  }
  if(ulNumBds < 1)
  {
      printf("  ERROR: No Boards found in the system\n");
      do{}while( !kbhit() );         // Wait for a Key Press
  }
  else
  {
    ulBdNum = 0;
    ClrScr();
    printf("\n\n");
    printf("  ====================================================\n");
    printf("   Select Board Number to Test                          \n");
    printf("  ====================================================\n");
    printf("%s\n\n",cBoardInfo);
    do
	{
      kbchar = prompt_for_key("  Please Make selection: \n");
          switch(kbchar)
          {
            case '1':
            {
              ulBdNum = 1;
              break;
            }
            case '2':
            {
              ulBdNum = 2;
              break;
            }
            case '3':
            {
              ulBdNum = 3;
              break;
            }
            case '4':
            {
              ulBdNum = 4;
              break;
            }
            case '5':
            {
              ulBdNum = 5;
              break;
            }
          }
     if((ulBdNum == 0) || (ulBdNum > ulNumBds))
       printf("  Error - This is not a valid entry");
	}while((ulBdNum == 0) || (ulBdNum > ulNumBds));

	if((ulErr = SIO4_Open(ulBdNum)) != 0)
	{
     ShowAPIError(ulErr);
     do{}while( !kbhit() );         // Wait for a Key Press
     exit(0);
	}
    ulAuxBdNum = 0; 
    if(ulNumBds >1){
      ulAuxBdNum = ulBdNum;
      ClrScr();
      printf("\n\n");
      printf("  ====================================================\n");
      printf("   Select Auxillary Board Number to Use               \n");
      printf("   For Examples Requiring Two Boards (0 = Disabled)   \n");
      printf("  ====================================================\n");
      printf("%s\n\n",cBoardInfo);
      do
	  {
        kbchar = prompt_for_key("  Please Make selection: \n");
          switch(kbchar)
          {
            case '0':
            {
              ulAuxBdNum = 0;
              break;
            }
            case '1':
            {
              ulAuxBdNum = 1;
              break;
            }
            case '2':
            {
              ulAuxBdNum = 2;
              break;
            }
            case '3':
            {
              ulAuxBdNum = 3;
              break;
            }
            case '4':
            {
              ulAuxBdNum = 4;
              break;
            }
            case '5':
            {
              ulAuxBdNum = 5;
              break;
            }
          }
        if((ulAuxBdNum == ulBdNum) || (ulAuxBdNum > ulNumBds))
       printf("  Error - This is not a valid entry");
	  }while((ulAuxBdNum == ulBdNum) || (ulBdNum > ulNumBds));
	}
	if(ulAuxBdNum)
		ulErr = SIO4_Open(ulAuxBdNum);
    if(ulErr)
	{
     ShowAPIError(ulErr);
     printf("\n\nAux Board Unavailable for Use\n");
     do{}while( !kbhit() );         // Wait for a Key Press
     ulAuxBdNum = 0;
	}

    do
    {
      ClrScr();
      printf("\n\n");
      printf("  ====================================================\n");
      printf("   SIO4 Sample Code - Main Board # %d                 \n",ulBdNum);
      if(ulAuxBdNum)
		  printf("                     Aux  Board # %d                 \n",ulAuxBdNum);
      printf("  ====================================================\n");
      printf("   1 - Reset Board                                    \n");
      printf("   2 - Write Local Register                           \n");
      printf("   3 - Read Local Register                            \n");
      printf("   4 - Write USC Register                             \n");
      printf("   5 - Read USC Register                              \n");
      printf("   6 - Internal Loopback Test                         \n");
	  printf("   X - EXIT (return to DOS)                           \n");
      printf("  ----------------------------------------------------\n");
      mmchar = prompt_for_key("  Please Make selection: \n");

      switch(mmchar)
      {
        case '1': //
           ResetBoard();
          break;
        case '2': //
           WriteLocal();
          break;
        case '3': //
           ReadLocal();
          break;
        case '4': //
           WriteUSC();
          break;
        case '5': //
           ReadUSC();
          break;
        case '6': //
           InternalLoopback();
          break;
        case 'X': //
          break;
        default:
          printf("  Invalid Selection: Press AnyKey to continue...\n");
          getch();
          break;
      }
    }while(mmchar!='X');

	SIO4_Close(ulBdNum);
    if(ulAuxBdNum)
	  SIO4_Close(ulAuxBdNum);
	// release memory
	if (hWriteGlobal){GlobalUnlock(hWriteGlobal);}
	if (pWriteBuffer){GlobalFree(hWriteGlobal);}
	if (hReadGlobal){GlobalUnlock(hReadGlobal);}
	if (pReadBuffer){GlobalFree(hReadGlobal);}

	CursorVisible(TRUE);
  }
} /* end main */

//------------------------------------------------------------------------------
void ResetBoard(void)
//------------------------------------------------------------------------------
{
  ClrScr();
  CurX=CurY=2;
  PositionCursor(CurX,CurY++);
  cprintf("Resetting All Channels");

  // Init the Board
  PositionCursor(CurX,CurY++);
  ulErr = SIO4_Board_Reset(ulBdNum);
  if(ulErr)
  {
   ShowAPIError(ulErr);
   do{}while( !kbhit() );         // Wait for a Key Press
   return;
  }
  Sleep(1000);
}

//------------------------------------------------------------------------------
void WriteLocal(void)
//------------------------------------------------------------------------------
{
  U32 ulRegister;
  U32 ulValue;

  ClrScr();
  CurX=CurY=2;
  PositionCursor(CurX,CurY++);
  cprintf(" 1. Write Local");
  PositionCursor(CurX,CurY++);
  cprintf(" Enter Register 99 to return to menu");
  PositionCursor(CurX,CurY++);
  PositionCursor(CurX,CurY++);
  do
  {
    PositionCursor(CurX,CurY);
    cprintf("Enter Register(Hex) to Write [0 - FC] ");
	scanf("%x",&ulRegister);
	if(ulRegister == 0x99)
		continue;
    PositionCursor(CurX,(U8)(CurY+1));
	if(ulRegister > 0xFC){
      cprintf("Invalid Register                 ");
	  continue;
	}
    cprintf("Enter 32Bit Value(Hex) to Write ");
	scanf("%x",&ulValue);
	SIO4_WriteLocalRegister(ulBdNum, ulRegister, ulValue);

    PositionCursor(CurX,CurY);
    cprintf("                                           ");
  }while(ulRegister != 0x99);
}

//------------------------------------------------------------------------------
void ReadLocal(void)
//------------------------------------------------------------------------------
{
  U32 ulRegister;

  ClrScr();
  CurX=CurY=2;
  PositionCursor(CurX,CurY++);
  cprintf(" 2. Read Local");
  PositionCursor(CurX,CurY++);
  cprintf(" Enter Register 99 to return to menu");
  PositionCursor(CurX,CurY++);
  PositionCursor(CurX,CurY++);
  do
  {
    PositionCursor(CurX,CurY);
    cprintf("Enter Register(Hex) to Read [0 - FC] ");
	scanf("%x",&ulRegister);
	if(ulRegister == 0x99)
		continue;
    PositionCursor(CurX,(U8)(CurY+1));
	if(ulRegister > 0xFC){
      cprintf("Invalid Register                       ");
	  continue;
	}
    cprintf("Register %02X Value is %08X",ulRegister,SIO4_ReadLocalRegister(ulBdNum, ulRegister, &ulErr));
    PositionCursor(CurX,CurY);
    cprintf("                                         ");

  }while(ulRegister != 0x99);
}

//------------------------------------------------------------------------------
void WriteUSC(void)
//------------------------------------------------------------------------------
{
  U32 ulChannel;
  U32 ulRegister;
  U32 ulValue;

  ClrScr();
  CurX=CurY=2;
  PositionCursor(CurX,CurY++);
  cprintf(" 3. Write USC");
  PositionCursor(CurX,CurY++);
  cprintf(" Enter Channel 0 to return to menu");
  PositionCursor(CurX,CurY++);
  PositionCursor(CurX,CurY++);
  do
  {
    PositionCursor(CurX,CurY);
    cprintf("Enter Channel to Write [1-4] ");
	scanf("%ld",&ulChannel);
	if(ulChannel == 0)
		continue;
    PositionCursor(CurX,(U8)(CurY+1));
	if(ulChannel > 4){
      cprintf("Invalid Channel                          ");
	  continue;
	}
    cprintf("Enter Register(Hex) to Write [0 - 3E] ");
	scanf("%lx",&ulRegister);
    PositionCursor(CurX,(U8)(CurY+2));
	if(ulRegister > 0x3E){
      cprintf("Invalid Register                         ");
	  continue;
	}
    cprintf("Enter 16Bit Value(Hex) to Write ");
	scanf("%lx",&ulValue);
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, ulRegister, ulValue);
    PositionCursor(CurX,CurY);
    cprintf("                                            ");
  }while(ulChannel != 0);
}

//------------------------------------------------------------------------------
void ReadUSC(void)
//------------------------------------------------------------------------------
{
  U32 ulChannel;
  U32 ulRegister;

  ClrScr();
  CurX=CurY=2;
  PositionCursor(CurX,CurY++);
  cprintf(" 4. Read USC");
  PositionCursor(CurX,CurY++);
  cprintf(" Enter Channel 0 to return to menu");
  PositionCursor(CurX,CurY++);
  PositionCursor(CurX,CurY++);
  do
  {
    PositionCursor(CurX,CurY);
    cprintf("Enter Channel to Read [1-4] ");
	scanf("%ld",&ulChannel);
	if(ulChannel == 0)
		continue;
    PositionCursor(CurX,(U8)(CurY+1));
	if(ulChannel > 4){
      cprintf("Invalid Channel                  ");
	  continue;
	}
    cprintf("Enter Register(Hex) to Read ");
	scanf("%lx",&ulRegister);
    PositionCursor(CurX,(U8)(CurY+2));
	if(ulRegister > 0x3E){
      cprintf("Invalid Register                     ");
	  continue;
	}
    cprintf("Register %02lX Value is %04lX",ulRegister,SIO4_ReadUSCRegister(ulBdNum, ulChannel, ulRegister, &ulErr));
    PositionCursor(CurX,CurY);
    cprintf("                                            ");
  }while(ulChannel != 0);
}


//------------------------------------------------------------------------------
void InternalLoopback(void)
//------------------------------------------------------------------------------
{

  U32 ulChannel;

  // Init the Board
  ulErr = SIO4_Board_Reset(ulBdNum);
  if(ulErr)
  {
   ShowAPIError(ulErr);
   do{}while( !kbhit() );         // Wait for a Key Press
   return;
  }
  kbflush();
  do
  {
    ClrScr();
    CurX=CurY=1;
    PositionCursor(CurX,CurY++);
    cprintf(" 6. Single Channel Internal Loopback Test (No Loopback Cable Required)");
    PositionCursor(CurX,CurY++);
    cprintf("    Enter X to return to Menu");
    PositionCursor(CurX,CurY++);
    PositionCursor(CurX,CurY++);
    cprintf(" Select Channel to test (1..4) ");

    kbchar=toupper(getch());
    if(kbchar != 'X'){
		if((kbchar > '0') && (kbchar < '5')){
		   ulChannel = kbchar - '0';

           SingleChannelInternalLoopbackTest(ulChannel);
		}

	}
  }while(kbchar != 'X');

}  //  end of InternalLoopback()

//////////////////////////////////////////////////////////////////////
//
//	Begin Test Functions:
//
//////////////////////////////////////////////////////////////////////

void SingleChannelInternalLoopbackTest(U32 ulChannel)
{

	U32	ulLoop;
	HANDLE myHandle;
	DWORD EventStatus;


	// Fill WriteBuffer and ReadBuffer 
    // Puts a count pattern in ReadBuffer
    // Puts a pattern in WriteBuffer of As, 5s, 0s, Fs, Count
	for (ulLoop = 0; ulLoop < ulBufferSize; ulLoop++)
	{
		pReadBuffer[ulLoop] = ulLoop;
		switch(ulLoop % 5)
		{
         case 0:{pWriteBuffer[ulLoop] = 0xAA;}break;
         case 1:{pWriteBuffer[ulLoop] = 0x55;}break;
         case 2:{pWriteBuffer[ulLoop] = 0x00;}break;
         case 3:{pWriteBuffer[ulLoop] = 0xFF;}break;
         case 4:{pWriteBuffer[ulLoop] = (int)ulLoop/5;}break;
	  } // end switch(remainder)
	} // end for 


	///////////////////////////////////////////////////////
	// Begin setup of SIO4 Registers
	///////////////////////////////////////////////////////

	// local offset 0x0: fw rev reg 
	// this is RO, so we will just display it
	
	cprintf("\n\tFirmware Revision:\t\t%08lX", SIO4_ReadLocalRegister(ulBdNum, FW_REV, &ulErr));

	// local offset 0x4: bd control reg 
	// used to assign demand mode DMA channel requests
	// since we are not using any DMA in these tests, we will just set it to 0x0
	SIO4_WriteLocalRegister(ulBdNum, BCR, 0x0);
		
	// local offset 0x8: reserved register on all early SIO4 models
	// the PCI-SIO4A and PMC-SIO4AR use this as a RO status register
	SIO4_WriteLocalRegister(ulBdNum, BSR, 0x0);
	
	// local offset 0xC: clock control reg
	// this register is used to enable the clock transceivers for all
	// four channels - this allows separate direction control of clock and data
	// for internal loopback, we will leave this register set to 0x0
	SIO4_WriteLocalRegister(ulBdNum, CLK_CONTROL, 0x0);
	
	// local offset 0x10/0x20/0x30/0x40: channel X Tx FIFO almost flags reg
	// this register is used to set the trigger point for the transmit FIFO almost
	// full and empty flags which can be read from the chan X control status
	// bits D0..15 = almost empty value, bits 16..31 = almost full value
	// for this example we will set each to 16
	SIO4_WriteLocalRegister(ulBdNum, ((ulChannel<<4) | TX_ALMOST), 0x00100010);

	// local offset 0x14/0x24/0x34/0x44: channel X Rx FIFO almost flags reg
	// this register is used to set the trigger point for the receive FIFO almost
	// full and empty flags which can be read from the chan X control status
	// bits D15..0 = almost empty value, bits 31..16 = almost full value
	// for this example we will set each to 16
	SIO4_WriteLocalRegister(ulBdNum, ((ulChannel<<4) | RX_ALMOST), 0x00100010);

	// local offset 0x18/0x28/0x38/0x48: channel X FIFO 
	// a write to this location will store data in the transmit FIFO
	// a read from this location will fetch data from the receive FIFO
	// at this point in the test, we will not write/read to/from the FIFO
	//SIO4_WriteLocalRegister(ulBdNum, ((ulChannel<<4) | FIFO), 0x0);

	// local offset 0x1C/0x2C/0x3C/0x4C: channel X control/status reg
	// this multi-purpose register performs the following control functions:
	// 1. reset transmit and receive FIFOs
	// 2. reset the Zilog chip (not always recommended as both channel in the 
	//    chip are cleared)
	// 3. enable data transceiver directions and upper or lower cable portions
	// this multi-purpose register performs the following status functions:
	// 1. status of the transmit FIFO can be read from bits D8..15
	// 2. status of the receive FIFO can be read from bits D16..31
	// since this test is internal loopback, we will just set to 0x3 to reset FIFOs
	SIO4_WriteLocalRegister(ulBdNum, ((ulChannel<<4) | CONTROL_STATUS), 0x0003);
	Sleep(10);

	// local offset 0x50/0x54/0x58/0x5C: channel X sync detect byte
	// can be used to trigger an interrupt when a specified byte is loaded
	// in to the receive FIFO - bits D0..8 are used to set the sync byte
	// not used for this test so we leave it set to 0x0
	SIO4_WriteLocalRegister(ulBdNum, SYNC_CHARACTER + ulChannel*4, 0x0);
	
	// local offset 0x60: interrupt control register
	// used to enable individual interrupt sources with a b'1' enabling the 
	// source and a b'0' disabling it - the source must be enabled before 
	// an interrupt will occur
	// not used for this test so we leave it set to 0x0
	SIO4_WriteLocalRegister(ulBdNum, ICR, 0x0);

	// local offset 0x64: interrupt status/clear register
	// used to show the status individual interrupt sources or to clear a 
	// pending interrupt by writing a b'1' to the respective bit
	// not used for this test so we leave it set to 0x0
	SIO4_WriteLocalRegister(ulBdNum, ISR, 0x0);

	// local offset 0x68: interrupt edge/level register
	// used to set individual interrupt sources as edge or level triggered
	// not used for this test so we leave it at default
	// SIO4_WriteLocalRegister(ulBdNum, INTR_EDGE_LEVEL, 0x0);

	// local offset 0x68: interrupt high/low register
	// used to set individual interrupt sources as active high or active low
	// not used for this test so we leave it at default
	// SIO4_WriteLocalRegister(ulBdNum, INTR_HIGH_LOW, 0x0);
	
	///////////////////////////////////////////////////////
	// Begin setup of USC Registers
	//
	// NOTE:
	// since there are so many registers located in the Zilog Z16C30 serial 
	// controller chips and so many functions are controlled with each reg,
	// we will only describe the functions we are configuring/changing
	// see the Z16C30 User Manual and Product Specification Databook for more 
	// info and definitions of each reg/bit
	//
	///////////////////////////////////////////////////////

	// local offset (channel base + 0x00/0x01): chan command/address reg
	// for internal loopback, we set mode to internal loopback
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, CHANNEL_COMMAND_ADDRESS, 0x0300);
	
	// local offset (channel base + 0x02/0x03): chan mode reg
	// for internal loopback, we set protocol to async/16x sampling/1 stop bit
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, CHANNEL_MODE, 0x0000);

	// local offset (channel base + 0x04/0x05): chan command/status reg
	// not used for this test - leave set at 0x0 (some RO bits)
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, CHANNEL_COMMAND_STATUS, 0x0000);

	// local offset (channel base + 0x06/0x07): chan control reg
	// not used for this test - leave set at 0x0 (some RO bits)
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, CHANNEL_CONTROL, 0x0000);

	// local offset (channel base + 0x0C/0x0D): test mode data reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TEST_MODE_DATA, 0x0000);

	// local offset (channel base + 0x0E/0x0F): test mode control reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TEST_MODE_CONTROL, 0x0000);
	
	// local offset (channel base + 0x10/0x11): clock mode control reg
	// this register defines the clock sources used for the serial channel
	// set receive clock source as baud rate generator 0 output
	// set transmit clock source as baud rate generator 0 output
	// set baud rate generator 0 source as Zilog /TxC pin
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, CLOCK_MODE_CONTROL, 0x0324);

	// local offset (channel base + 0x12/0x13): hardware config reg
	// this register sets BRG functions as well as DMA config
	// enable baud rate generator 0 
	// set /RxACK pin control to Rx Acknowledge input
	// set /TxACK pin control to Tx Acknowledge input
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, HARDWARE_CONFIGURATION, 0x0045);

	// local offset (channel base + 0x14/0x15): interrupt vector reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, INTERRUPT_VECTOR, 0x0000);

	// local offset (channel base + 0x16/0x17): IO control reg
	// this register sets various pin functions of the serial channel
	// set /RxC pin as an input
	// set /TxC pin as an input
	// set /TxD pin as transmit data output
	// set /RxREQ pin control to Rx Request output
	// set /TxREQ pin control to Tx Request output
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, IO_CONTROL, 0x0F00);

	// local offset (channel base + 0x18/0x19): interrupt control reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, INTERRUPT_CONTROL, 0x0);

	// local offset (channel base + 0x1A/0x1B): daisy chain ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, DAISY_CHAIN_CONTROL, 0x0);

	// local offset (channel base + 0x1C/0x1D): misc int ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, MISC_INTERRUPT_STATUS, 0x0);

	// local offset (channel base + 0x1E/0x1F): stat int ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, STATUS_INTERRUPT_CONTROL, 0x0);

	// local offset (channel base + 0x20): receive data reg
	// can be used to read data directly from Z16C30 internal FIFOs
	// not used for this test - leave set at 0x0
	// SIO4_ReadUSCRegister(ulBdNum, ulChannel, RECEIVE_DATA, &ulErr);

	// local offset (channel base + 0x22/0x23): receive mode reg
	// sets up the mode of the serial channel receiver
	// set Rx enable to enable without auto enables
	// Rx char length 8 bits
	// Rx parity even
	// Rx data encoding NRZ
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_MODE, 0x0002);

	// local offset (channel base + 0x24/0x25): receive cmd/stat reg
	// not used for this test - leave set at 0x0
	//SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_COMMAND_STATUS, 0x0);

	// local offset (channel base + 0x26/0x27): receive int ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_INTERRUPT_CONTROL, 0x0);

	// local offset (channel base + 0x28/0x29): receive int ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_SYNC, 0x0);

	// local offset (channel base + 0x2A/0x2B): receive cnt limit reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_COUNT_LIMIT, 0x0);

	// local offset (channel base + 0x2C/0x2D): receive char cnt reg
	// not used for this test - leave set at 0x0 - RO
	// SIO4_WriteUSCRegister(ulBdNum, ulChannel, RECEIVE_CHARACTER_COUNT, 0x0);
	
	// local offset (channel base + 0x2E/0x2F): time constant 0 reg
	// this reg is used to set the time constant for baud rate generator 0
	// output clock = inputclock/(TC0_value + 1)
	// so we use a value of 19 to divide onboard 20MHz/20 = 1MHz
	// set clock speed to 1MHz - with 16x sampling this gives us an
	// actual baud rate of about 62.5 Kbaud
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TIME_CONSTANT_0, 0x0013);
	
	// local offset (channel base + 0x30): transmit data reg
	// can be used to write data directly from Z16C30 internal FIFOs
	// not used for this test - leave set at 0x0
	// SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_DATA, 0x0);

	// local offset (channel base + 0x32/0x33): transmit mode reg
	// sets up the mode of the serial channel transmitter
	// set Tx enable to enable without auto enables
	// Tx char length 8 bits
	// Tx parity disabled
	// Tx parity even
	// Tx data encoding NRZ
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_MODE, 0x0002);

	// local offset (channel base + 0x34/0x35): transmit cmd/stat reg
	// not used for this test - leave set at 0x0
	// SIO4_WriteLocalRegister(ulBdNum, TRANSMIT_COMMAND_STATUS, 0x0);
	
	// local offset (channel base + 0x36/0x37): transmit int ctrl reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_INTERRUPT_CONTROL, 0x0);

	// local offset (channel base + 0x38/0x39): transmit sync reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_SYNC, 0x0);
	
	// local offset (channel base + 0x3A/0x3B): transmit cnt limit reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_COUNT_LIMIT, 0x0);
	
	// local offset (channel base + 0x3C/0x3D): transmit char cnt reg
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TRANSMIT_CHARACTER_COUNT, 0x0);
	
	// local offset (channel base + 0x3E/0x3F): TC1
	// not used for this test - leave set at 0x0
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, TIME_CONSTANT_1, 0x0);
	
    if(ulErr = SIO4_Open_DMA_Channel(ulBdNum, 0)){ // Use DMA Channel 0
		   ShowAPIError(ulErr);
		   return;
	}


	////////////////////////////////////////////////////////////////////////
	//	setup complete, ready to test data xfer
	////////////////////////////////////////////////////////////////////////
	// load data from WriteBuffer in to FIFO
    if(ulErr = SIO4_DMA_To_FIFO(ulBdNum,ulChannel,0,ulBufferSize,pWriteBuffer)){
		   ShowAPIError(ulErr);
	       SIO4_Close_DMA_Channel(ulBdNum, 0);
		   return;
	}


    SIO4_Attach_Interrupt(ulBdNum,&myHandle,Local,(2<<((ulChannel-1)*4)));

    // We have data loaded to the FIFO, now enable the USC to
	// transfer the data and do the loopback.
	// We have to do it this way to see the interrupt on the older 
	// versions of the board because they are level sensitive,
	// not edge triggered.
	SIO4_WriteUSCRegister(ulBdNum, ulChannel, IO_CONTROL, 0x0500);

	/* We could have used PIO mode and loaded the FIFO ourselves
	for (ulLoop = 0; ulLoop < ulBufferSize; ulLoop++)
		SIO4_WriteLocalRegister(ulBdNum, ((ulChannel<<4) | FIFO), *(pWriteBuffer + ulLoop) );
    */
   EventStatus = WaitForSingleObject(myHandle,1 * 1000); // Wait for the interrupt

   switch(EventStatus)
   {
	case WAIT_OBJECT_0:
		cprintf("\n\tInterrupt was requested    ...");
		break;
	default:
		cprintf("\n\tInterrupt was NOT requested...");
		break;
   }

    /* We would use the following if using PIO mode
	// poll on the Tx FIFO to go empty	
	// !kbhit() gives an out if something goes wrong
	while(((ulValue = SIO4_ReadLocalRegister(ulBdNum, CONTROL_STATUS, &ulErr)) & 0x100) && !kbhit() )
		cprintf("\n\tchannel %d status --> %X", ulChannel, ulValue);
	Sleep(10);
    */

	Sleep(100);
   
	cprintf("\n\n\tByte\tRead\tExpected");
	cprintf("\n\tNumber:\tValue:\tValue:");

    if(ulErr = SIO4_DMA_From_FIFO(ulBdNum,ulChannel,0,ulBufferSize,pReadBuffer)){
		   ShowAPIError(ulErr);
	       SIO4_Close_DMA_Channel(ulBdNum, 0);
		   return;
	}
	
	
	/* We would use this to fetch the data in PIO mode
	// fetch data from FIFO and load in to ReadBuffer
	for (ulLoop = 0; ulLoop < ulBufferSize; ulLoop++)
		*(pReadBuffer + ulLoop) = SIO4_ReadLocalRegister(ulBdNum, ((ulChannel<<4) | FIFO), &ulErr)&0xFF;
	*/
	
	// Only display the first 16 words
	for (ulLoop = 0; ulLoop < ((ulBufferSize < 0x10)?ulBufferSize:0x10); ulLoop++)
		cprintf("\n\t%d\t%02X\t%02X", ulLoop, (*(pReadBuffer + ulLoop)&0xFF), *(pWriteBuffer + ulLoop));

	// print status
	cprintf("\n\tCONTROL_STATUS:\t\t%08lX", SIO4_ReadLocalRegister(ulBdNum, CONTROL_STATUS + (ulChannel << 4) , &ulErr));
	
	// dump contents of FIFO is not empty
//	while((SIO4_ReadLocalRegister(ulBdNum, CONTROL_STATUS + (ulChannel << 4) , &ulErr)) & 0x1000)
//		cprintf("\n\tData:\t\t%08lX", SIO4_ReadLocalRegister(ulBdNum, FIFO + (ulChannel << 4) , &ulErr));




	SIO4_Close_DMA_Channel(ulBdNum, 0);

    anykey();
} // end SingleChannelInternalLoopbackTest()

